#version 130

/*=========================================================================== \\
||  "PPSSPP Multi-Pass Shaders" by Project Kube <http://project-kube.de/> is  ||
||       licensed under the Creative Commons Attribution-ShareAlike 4.0       ||
||        International License. To view a copy of this license, visit        ||
||                http://creativecommons.org/licenses/by-sa/4.0/              ||
\\============================================================================*/

// ------ CONFIGURATION ------
	// Anti-Alias (FXAA)
#define ENABLE_ANTIALIAS		// Enable Anti-Aliasing
const float	AA_Span_Max			= 5.0;		// Antialias maximum span size.
const float	AA_Reduce_Mul		= 1.0/AA_Span_Max;	// Unsure
const float	AA_Reduce_Min		= (1.0/128.0);		// Unsure

	// Sharpen
#define ENABLE_SHARPEN			// Enable Sharpening
const float Sharpen_Kernel[]	= float[9](	// Kernel for sharpening. (Normally left alone)
									 0,-1, 0,
									-1, 5,-1,
									 0,-1, 0);
const float Sharpen_KernelSpan	= 1.00f;	// Kernel texel span. (Range 1..Inf)
const float Sharpen_Multiply	= 1.00f;	// Multiply texel colors by this amount. (Range 0..Inf)
const float	Sharpen_Strength	= 0.20f;	// How strongly visible is the sharpen effect? (Range 0..1)

	// Bloom
#define ENABLE_BLOOM			// Enable Bloom
const int	Bloom_Kernel		= 5;		// Kernel size to use for Bloom. (Range 1..Inf)
const float Bloom_KernelSpan	= 1.0f;		// Kernel texel span. (Range 1..Inf)
const float Bloom_Multiplier	= 1.0f;		// Multiplier for final value. (Range 0..Inf)
const float	Bloom_Strength		= 1.0f;		// How strongly visible is the bloom effect? (Range 0..1)
	// Color Adjustments for bloom layer.
const float Bloom_Hue			= 0.00f;	// Shift Hue by this amount. (Range -1..1)
const float Bloom_Saturation	= 0.33f;	// Adjust Saturation by this amount. (Range 0..2, 1 is Neutral)
const float Bloom_Value			= 1.00f;	// Adjust Value by this amount. (Range 0..2, 1 is Neutral)
const float	Bloom_Contrast		= 4.00f;	// Contrast amount applied. (Range 0..Inf, 1 is Neutral)
const float	Bloom_Brightness	= -1.95f;	// Brightness added, 0.0 is neutral. (Range -1..1)
	// Bloom Blending Mode
#define BLOOM_ADDITIVE	// Strong Effect, High Overexposure
//#define BLOOM_SCREEN	// Medium Effect, Medium Overexposure
//#define BLOOM_DEBUG	//Debugging mode, shows v4BloomFinal

	// Color Manipulation
#define ENABLE_COLORMANIPULATION	// Enable Color manipulation
const float	CM_Hue				= 0.00f;	// Shift Hue by this amount. (Range -1..1)
const float	CM_Saturation		= 1.33f;	// Adjust Saturation by this amount. (Range 0..2, 1 is Neutral)
const float	CM_Value			= 1.25f;	// Adjust Value by this amount. (Range 0..2, 1 is Neutral)
const float	CM_Contrast			= 1.125f;	// Contrast amount applied. (Range 0..Inf, 1 is Neutral)
const float	CM_Brightness		= -0.025f;	// Brightness added, 0.0 is neutral. (Range -1..1)

	// Vignette
#define ENABLE_VIGNETTE			// Enable Vignette
const float Vignette_Strength	= 0.33f; // How visible is the vignette effect? (Range 0..1)

	// Scanlines
//#define ENABLE_SCANLINES		// Enable Scanlines
const float SL_CrawlSpeed		= 1.00f;	// Speed of scanline crawling.
const float SL_MinIntensity		= 0.75f;	// Minimum intensity (dark lines).
const float SL_MaxIntensity		= 1.00f;	// Maximum intensity (bright lines).
const float SL_Strength			= 1.00f;	// Strength of scanline effect.
#define SL_NTSCBLEEDING			// Enable NTSC color bleeding?

	// Curvature
//#define ENABLE_CURVATURE		// Enable curvature
const float Curvature_Strength	= 0.33f;	// Curvature strength, 0.33 is newer generation CRT, 1.00 is really old CRT.

// ------ ------ ------ ------ CODE ------ ------ ------ ------
#ifdef GL_ES // Medium Precision for ES-Platforms.
	precision mediump float;
	precision mediump int;
#else	// High Precision for non-ES-Platforms.
	precision highp float;
	precision highp int;
#endif

	// Math - PI(e)
#define PI		3.1415926f
#define TwoPI	6.2831853f
#define HalfPI	1.5707963f

	// Input - PPSSPP
uniform sampler2D sampler0;
uniform vec2 u_texelDelta;		// vec2(1.0 / renderWidth, 1.0 / renderHeight)
uniform vec2 u_pixelDelta;		// Respects "AtOutputResolution" option, when inactive same as u_texelDelta, otherwise: vec2(1.0 / 480, 1.0 / 272)
uniform vec4 u_time;			// vec4(TimeInSecondsSinceStart, ((Frame % 60) / 60.0), Frame, ActualFrame % 60)

	// Input - Resolution
vec2 c_resolution = 1.0f / u_texelDelta;
vec2 c_outputresolution = 1.0f / u_pixelDelta;
vec2 c_resolutionScale = c_resolution / vec2(480.0f, 272.0f);

	// Input - Texel Min Max
vec2 c_texelMin = vec2(0, u_texelDelta.y * (c_resolutionScale.y + 0.5f));
vec2 c_texelMax = vec2(1.0f - u_texelDelta.x * (c_resolutionScale.x + 0.5f), 1);

	// Input - Vertex Shader
in vec2 v_v2TexCoord;

	// Output - Fragment Buffer
out vec4 v_v4FragColor;

// --------------------------------------------- Functions
	// Functions - Convert RGBA To HSVA
#define RGBToHSV_Optimized
vec3 RGBToHSV(vec3 RGB) {
	const vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);
	const float e = 1.0e-10;
	
	#ifndef RGBToHSV_Optimized
		vec4 p = mix(vec4(RGB.bg, K.wz), vec4(RGB.gb, K.xy), step(RGB.b, RGB.g));
		vec4 q = mix(vec4(p.xyw, RGB.r), vec4(RGB.r, p.yzx), step(p.x, RGB.r));
	#else
		vec4 p = RGB.g < RGB.b ? vec4(RGB.bg, K.wz) : vec4(RGB.gb, K.xy);
		vec4 q = RGB.r < p.x ? vec4(p.xyw, RGB.r) : vec4(RGB.r, p.yzx);
	#endif
	
	float d = q.x - min(q.w, q.y);
	return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
}

	// Functions - Convert HSVA To RGBA
const vec4 HSVToRGB_K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);

#define Inline_HSVToRGB
#ifndef Inline_HSVToRGB
	vec3 HSVToRGB(vec3 v3HSV) {
		return v3HSV.z * mix(HSVToRGB_K.xxx, clamp(abs(fract(v3HSV.xxx + HSVToRGB_K.xyz) * 6.0 - HSVToRGB_K.www) - HSVToRGB_K.xxx, 0.0, 1.0), v3HSV.y);
	}
#else
	#define HSVToRGB(HSVA) HSVA.z * mix(HSVToRGB_K.xxx, clamp(abs(fract(HSVA.xxx + HSVToRGB_K.xyz) * 6.0 - HSVToRGB_K.www) - HSVToRGB_K.xxx, 0.0, 1.0), HSVA.y)
#endif

	// Functions - Gaussian Distribution
#define Inline_Gaussian
#ifndef Inline_Gaussian
	float Gaussian (float x, float deviation) {
		return (1.0 / sqrt(2.0 * 3.141592 * deviation)) * exp(-((x * x) / (2.0 * deviation)));	
	}
#else
	#define Gaussian(x,deviation) (1.0 / sqrt(2.0 * 3.141592 * deviation)) * exp(-((x * x) / (2.0 * deviation)));
#endif

// --------------------------------------------- Pass: Diffuse
#define Inline_Pass_Diffuse
#ifndef Inline_Pass_Diffuse
	vec3 Pass_Diffuse(vec2 v2TexCoord) {
		return texture2D(sampler0, v2TexCoord).xyz;
	}
#else
	#define Pass_Diffuse(v2TexCoord) texture2D(sampler0, v2TexCoord).xyz
#endif
#define PREVIOUS_PASS Pass_Diffuse

// --------------------------------------------- Pass: Anti Alias
#ifdef ENABLE_ANTIALIAS
	vec3 Pass_AntiAlias(vec2 v2TexCoord) {
		/*vec3 rgbNW = PREVIOUS_PASS(v2TexCoord + vec2(-u_texelDelta.x, -u_texelDelta.y)).xyz;
		vec3 rgbNE = PREVIOUS_PASS(v2TexCoord + vec2( u_texelDelta.x, -u_texelDelta.y)).xyz;
		vec3 rgbSW = PREVIOUS_PASS(v2TexCoord + vec2(-u_texelDelta.x,  u_texelDelta.y)).xyz;
		vec3 rgbSE = PREVIOUS_PASS(v2TexCoord + vec2( u_texelDelta.x,  u_texelDelta.y)).xyz;
		vec4 rgbM = PREVIOUS_PASS(v2TexCoord);*/
		vec3 rgbNW = texture2D(sampler0, v2TexCoord + vec2(-u_texelDelta.x, -u_texelDelta.y)).xyz;
		vec3 rgbNE = texture2D(sampler0, v2TexCoord + vec2( u_texelDelta.x, -u_texelDelta.y)).xyz;
		vec3 rgbSW = texture2D(sampler0, v2TexCoord + vec2(-u_texelDelta.x,  u_texelDelta.y)).xyz;
		vec3 rgbSE = texture2D(sampler0, v2TexCoord + vec2( u_texelDelta.x,  u_texelDelta.y)).xyz;
		vec4 rgbM = texture2D(sampler0, v2TexCoord);
		
		vec3 luma = vec3(0.299, 0.587, 0.114);
		float lumaNW = dot(rgbNW, luma);
		float lumaNE = dot(rgbNE, luma);
		float lumaSW = dot(rgbSW, luma);
		float lumaSE = dot(rgbSE, luma);
		float lumaM  = dot(rgbM.xyz, luma);
		
		float lumaMin = min(lumaM, min(min(lumaNW, lumaNE), min(lumaSW, lumaSE)));
		float lumaMax = max(lumaM, max(max(lumaNW, lumaNE), max(lumaSW, lumaSE)));
		
		vec2 dir;
		dir.x = -((lumaNW + lumaNE) - (lumaSW + lumaSE));
		dir.y =  ((lumaNW + lumaSW) - (lumaNE + lumaSE));
  
		float dirReduce = max((lumaNW + lumaNE + lumaSW + lumaSE) * (0.25 * AA_Reduce_Mul), AA_Reduce_Min);
		
		float rcpDirMin = 1.0/(min(abs(dir.x), abs(dir.y)) + dirReduce);
		
		dir = min(vec2(AA_Span_Max,  AA_Span_Max), 
		max(vec2(-AA_Span_Max, -AA_Span_Max), dir * rcpDirMin)) * u_texelDelta;

		vec3 rgbA = (1.0/2.0) * (
			PREVIOUS_PASS(v2TexCoord.xy + dir * (1.0/3.0 - 0.5)).xyz +
			PREVIOUS_PASS(v2TexCoord.xy + dir * (2.0/3.0 - 0.5)).xyz);
		vec3 rgbB = rgbA * (1.0/2.0) + (1.0/4.0) * (
			PREVIOUS_PASS(v2TexCoord.xy + dir * (0.0/3.0 - 0.5)).xyz +
			PREVIOUS_PASS(v2TexCoord.xy + dir * (3.0/3.0 - 0.5)).xyz);
		float lumaB = dot(rgbB, luma);
		
		if ((lumaB < lumaMin) || (lumaB > lumaMax)) {
			return rgbA;
		} else {
			return rgbB;
		}
	}
	
	#ifdef PREVIOUS_PASS
		#undef PREVIOUS_PASS
		#define PREVIOUS_PASS Pass_AntiAlias
	#endif
#endif

// --------------------------------------------- Pass: Sharpen
#ifdef ENABLE_SHARPEN
	vec3 Pass_Sharpen(vec2 v2TexCoord) {
		vec3 v3Sharpen = vec3(0);
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2(-1, -1) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[0] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2(-1,  0) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[1] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2(-1,  1) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[2] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2( 0, -1) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[3] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2( 0,  0) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[4] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2( 0,  1) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[5] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2( 1, -1) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[6] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2( 1,  0) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[7] * Sharpen_Multiply;
		v3Sharpen += PREVIOUS_PASS(clamp(v2TexCoord + vec2( 1,  1) * u_texelDelta * Sharpen_KernelSpan, c_texelMin, c_texelMax)).xyz * Sharpen_Kernel[8] * Sharpen_Multiply;
		return mix(PREVIOUS_PASS(v2TexCoord), v3Sharpen, Sharpen_Strength);
	}
	
	#ifdef PREVIOUS_PASS
		#undef PREVIOUS_PASS
		#define PREVIOUS_PASS Pass_Sharpen
	#endif
#endif

// --------------------------------------------- Pass: Bloom
#ifdef ENABLE_BLOOM
	vec3 Pass_Bloom(vec2 v2TexCoord) {
		// Gather Bloom Texels
		vec3 v3Bloom = vec3(0);
		const float gaussDeviation = (Bloom_Kernel * 0.35) * (Bloom_Kernel * 0.35);
			// This loop allows us to have more samples than we specify, by just taking the interpolated value.
		for (int x = -Bloom_Kernel; x < Bloom_Kernel; x++) { // Start at x=-Bloom_Kernel+0.5, Stop at x=Bloom_Kernel-1.5
			float gaussX = Gaussian(x, gaussDeviation);
			for (int y = -Bloom_Kernel; y < Bloom_Kernel; y++) { // Start at y=-Bloom_Kernel+0.5, Stop at x=Bloom_Kernel-1.5
				float gaussY = Gaussian(y, gaussDeviation);
				v3Bloom += texture2D(sampler0, clamp(v2TexCoord + vec2(x + 0.5, y + 0.5) * u_texelDelta * Bloom_KernelSpan, c_texelMin, c_texelMax)).xyz * gaussX * gaussY;
			}
		}
		
			// Apply Hue-, Saturation- and Value-shift
		vec3 v3HSV = RGBToHSV(v3Bloom);
		v3HSV.yz *= vec2(Bloom_Saturation, Bloom_Value);
		v3HSV.x += Bloom_Hue;
		
			// Apply Contrast and Brightness
		vec3 v3BloomFinal = clamp((((HSVToRGB(v3HSV) - 0.5f) * max(Bloom_Contrast, 0)) + 0.5f) + Bloom_Brightness, 0.0f, 1.0f) * Bloom_Multiplier;
		vec3 v3Input = PREVIOUS_PASS(v2TexCoord);
		
		#ifdef BLOOM_ADDITIVE
			return mix(v3Input, v3Input + v3BloomFinal, Bloom_Strength);
		#else
			#ifdef BLOOM_SCREEN
				return mix(v3Input, (v3Input + min(v3BloomFinal, 1.0f)) - (v3Input * min(v3BloomFinal, 1.0f)), Bloom_Strength);
			#else
				#ifdef BLOOM_DEBUG
					return v3BloomFinal;
				#else
					return v3Input;
				#endif
			#endif
		#endif
	}
	
	#ifdef PREVIOUS_PASS
		#undef PREVIOUS_PASS
		#define PREVIOUS_PASS Pass_Bloom
	#endif
#endif

// --------------------------------------------- Pass: Color Manipulation
#ifdef ENABLE_COLORMANIPULATION
	vec3 Pass_ColorManipulation(vec2 v2TexCoord) {
			// Apply Hue-, Saturation- and Value-shift
		vec3 v3HSV = RGBToHSV(PREVIOUS_PASS(v2TexCoord));
		v3HSV.x += CM_Hue;
		v3HSV.yz *= vec2(CM_Saturation, CM_Value);
		
			// Apply Contrast and Brightness
		return (((HSVToRGB(v3HSV) - 0.5f) * CM_Contrast) + 0.5f) + CM_Brightness;
	}
	
	#ifdef PREVIOUS_PASS
		#undef PREVIOUS_PASS
		#define PREVIOUS_PASS Pass_ColorManipulation
	#endif
#endif

// --------------------------------------------- Pass: Vignette
#ifdef ENABLE_VIGNETTE
	vec3 Pass_Vignette(vec2 v2TexCoord) {
		return PREVIOUS_PASS(v2TexCoord) * mix(1.0, 1.1 - 0.6 * (dot(v2TexCoord - 0.5, v2TexCoord - 0.5) * 2.0), Vignette_Strength);
	}
		
	#ifdef PREVIOUS_PASS
		#undef PREVIOUS_PASS
		#define PREVIOUS_PASS Pass_Vignette
	#endif
#endif

// --------------------------------------------- Pass: Scanlines
#ifdef ENABLE_SCANLINES
	vec3 Pass_Scanlines(vec2 v2TexCoord) {
		vec3 trueColor = PREVIOUS_PASS(v2TexCoord);
		
		// START OF: Code from crt.fsh
		// scanlines
		int vPos = int( ( v2TexCoord.y + u_time.x * 0.5 ) * 272.0 );
		float line_intensity = mod( float(vPos), 2.0 );

		#ifdef SL_NTSCBLEEDING
		// color shift
		float off = line_intensity * 0.0005;
		vec2 shift = vec2( off, 0 );

		// shift R and G channels to simulate NTSC color bleed
		vec2 colorShift = vec2( 0.001, 0 );
		float r = PREVIOUS_PASS( v2TexCoord + colorShift + shift ).x;
		float g = PREVIOUS_PASS( v2TexCoord - colorShift + shift ).y;
		float b = trueColor.z;
		
		vec3 c = vec3( r, g * 0.99, b ) * clamp( line_intensity, SL_MinIntensity, SL_MaxIntensity );
		#else
		vec3 c = trueColor * clamp( line_intensity, SL_MinIntensity, SL_MaxIntensity );
		#endif

		float rollbar = sin( ( v2TexCoord.y + u_time.x ) * 4.0 );
		// END OF: Code from crt.fsh
		
		return mix(trueColor, c + (rollbar * 0.02), SL_Strength);
	}
		
	#ifdef PREVIOUS_PASS
		#undef PREVIOUS_PASS
		#define PREVIOUS_PASS Pass_Scanlines
	#endif
#endif

// --------------------------------------------- Pass: Curvature
#ifdef ENABLE_CURVATURE
	vec3 Pass_Curvature(vec2 v2TexCoord) {
		// Bend texture coordinates
		vec2 cc = v2TexCoord - vec2(0.5, 0.5);
		float dist = dot(cc, cc) * Curvature_Strength;
		vec2 v2BentTexCoord = clamp(v2TexCoord + cc * (1.0 + dist) * dist, c_texelMin, c_texelMax);
		
		if ((v2BentTexCoord.x <= c_texelMin.x || v2BentTexCoord.x >= c_texelMax.x) ||
			(v2BentTexCoord.y <= c_texelMin.y || v2BentTexCoord.y >= c_texelMax.y)) {
			return vec3(0.0, 0.0, 0.0);
		} else {
			return PREVIOUS_PASS(v2BentTexCoord);
		}
	}
		
	#ifdef PREVIOUS_PASS
		#undef PREVIOUS_PASS
		#define PREVIOUS_PASS Pass_Curvature
	#endif
#endif

// --------------------------------------------- Pass: Final Result
void main() {
	v_v4FragColor.rgb = PREVIOUS_PASS(v_v2TexCoord);
	//gl_FragColor.rgb = vec3((1 / u_pixelDelta) / 272.0f / 5.0f, 0);
	v_v4FragColor.a = 1.0f;
} 